/*
 *  Shadow Maps
 *
 *  'm' Shadow map/Image(mode)
 *  'o' Objects
 *  +/- Light elevation
 *  []  Light position
 *  's' Start/stop light animation
 *  <>  Number of slices in objects
 *  'a' Toggle axes
 *  'b' Toggle box
 *  '0' snaps angles to 0,0
 *  arrows to rotate the world
 *  PgUp/PgDn zooms in/out
 */
#include "CSCIx229.h"
typedef struct {float x,y,z;} Point;
//  Global variables
int          mode=0;    // Display mode
int          obj=15;    // Display objects (bitmap)
int          move=1;    // Light movement
int          axes=1;    // Display axes
int          box=0;     // Display enclosing box
int          n=8;       // Number of slices
int          th=-30;    // Azimuth of view angle
int          ph=+30;    // Elevation of view angle
int          tex2d[3];  // Textures (names)
int          dt=50;     // Timer period (ms)
double       asp=1;     // Aspect ratio
double       dim=3;     // Size of world
int          zh=0;      // Light azimuth
float        Ylight=2;  // Elevation of light
float        Lpos[4];   // Light position
unsigned int framebuf=0;// Frame buffer id
double       Svec[4];   //  Texture planes S
double       Tvec[4];   //  Texture planes T
double       Rvec[4];   //  Texture planes R
double       Qvec[4];   //  Texture planes Q
int          Width;     //  Window width
int          Height;    //  Window height
int          Sdim;      //  Texture width
int          Tdim;      //  Texture height
int          shadowdim; //  Size of shadow map textures
int          NoAmbient; //  Enable ambient shadows
int          MultiTex;  //  Multiple textures
char* text[]={"Shadows","Shadow Map"};

#define MAXN 64    // Maximum number of slices (n) and points in a polygon

/*
 *  Get biggest power of two less than value
 */
int power2(int n)
{
   int i;
   for (i=1;2*i<=n;i*=2);
   return i;
}

/*
 *  Draw a cube
 */
static void Cube(float x,float y,float z , float th,float ph , float D)
{
   //  Transform
   glPushMatrix();
   glTranslated(x,y,z);
   glRotated(ph,1,0,0);
   glRotated(th,0,1,0);
   glScaled(D,D,D);

   glBegin(GL_QUADS);
   //  Front
   glColor3f(1,0,0);
   glNormal3f( 0, 0, 1);
   glTexCoord2f(0,0); glVertex3f(-1,-1, 1);
   glTexCoord2f(1,0); glVertex3f(+1,-1, 1);
   glTexCoord2f(1,1); glVertex3f(+1,+1, 1);
   glTexCoord2f(0,1); glVertex3f(-1,+1, 1);
   //  Back
   glColor3f(0,0,1);
   glNormal3f( 0, 0,-1);
   glTexCoord2f(0,0); glVertex3f(+1,-1,-1);
   glTexCoord2f(1,0); glVertex3f(-1,-1,-1);
   glTexCoord2f(1,1); glVertex3f(-1,+1,-1);
   glTexCoord2f(0,1); glVertex3f(+1,+1,-1);
   //  Right
   glColor3f(1,1,0);
   glNormal3f(+1, 0, 0);
   glTexCoord2f(0,0); glVertex3f(+1,-1,+1);
   glTexCoord2f(1,0); glVertex3f(+1,-1,-1);
   glTexCoord2f(1,1); glVertex3f(+1,+1,-1);
   glTexCoord2f(0,1); glVertex3f(+1,+1,+1);
   //  Left
   glColor3f(0,1,0);
   glNormal3f(-1, 0, 0);
   glTexCoord2f(0,0); glVertex3f(-1,-1,-1);
   glTexCoord2f(1,0); glVertex3f(-1,-1,+1);
   glTexCoord2f(1,1); glVertex3f(-1,+1,+1);
   glTexCoord2f(0,1); glVertex3f(-1,+1,-1);
   //  Top
   glColor3f(0,1,1);
   glNormal3f( 0,+1, 0);
   glTexCoord2f(0,0); glVertex3f(-1,+1,+1);
   glTexCoord2f(1,0); glVertex3f(+1,+1,+1);
   glTexCoord2f(1,1); glVertex3f(+1,+1,-1);
   glTexCoord2f(0,1); glVertex3f(-1,+1,-1);
   //  Bottom
   glColor3f(1,0,1);
   glNormal3f( 0,-1, 0);
   glTexCoord2f(0,0); glVertex3f(-1,-1,-1);
   glTexCoord2f(1,0); glVertex3f(+1,-1,-1);
   glTexCoord2f(1,1); glVertex3f(+1,-1,+1);
   glTexCoord2f(0,1); glVertex3f(-1,-1,+1);
   glEnd();

   // Restore
   glPopMatrix();
}

/*
 *  Draw a cylinder
 */
static void Cylinder(float x,float y,float z , float th,float ph , float R,float H)
{
   int i,j;   // Counters
   int N=4*n; // Number of slices

   //  Transform
   glPushMatrix();
   glTranslated(x,y,z);
   glRotated(ph,1,0,0);
   glRotated(th,0,1,0);
   glScaled(R,R,H);
   glColor3f(0,1,1);

   //  Two end caps (fan of triangles)
   for (j=-1;j<=1;j+=2)
   {
      glNormal3d(0,0,j); 
      glBegin(GL_TRIANGLE_FAN);
      glTexCoord2d(0,0); glVertex3d(0,0,j);
      for (i=0;i<=N;i++)
      {
         float th = j*i*360.0/N;
         glTexCoord2d(Cos(th),Sin(th)); glVertex3d(Cos(th),Sin(th),j);
      }
      glEnd();
   }

   //  Cylinder Body (strip of quads)
   glBegin(GL_QUADS);
   for (i=0;i<N;i++)
   {
      float th0 =  i   *360.0/N;
      float th1 = (i+1)*360.0/N;
      glNormal3d(Cos(th0),Sin(th0),0); glTexCoord2d(0,th0/90.0); glVertex3d(Cos(th0),Sin(th0),+1);
      glNormal3d(Cos(th0),Sin(th0),0); glTexCoord2d(2,th0/90.0); glVertex3d(Cos(th0),Sin(th0),-1);
      glNormal3d(Cos(th1),Sin(th1),0); glTexCoord2d(2,th1/90.0); glVertex3d(Cos(th1),Sin(th1),-1);
      glNormal3d(Cos(th1),Sin(th1),0); glTexCoord2d(0,th1/90.0); glVertex3d(Cos(th1),Sin(th1),+1);
   }
   glEnd();

   //  Restore
   glPopMatrix();
}

/*
 *  Draw torus
 */
static void Torus(float x,float y,float z , float th,float ph , float S,float r)
{
   int i,j;   // Counters
   int N=4*n; // Number of slices

   //  Transform
   glPushMatrix();
   glTranslated(x,y,z);
   glRotated(ph,1,0,0);
   glRotated(th,0,1,0);
   glScaled(S,S,S);
   glColor3f(1,1,0);

   //  Loop along ring
   glBegin(GL_QUADS);
   for (i=0;i<N;i++)
   {
      float th0 =  i   *360.0/N;
      float th1 = (i+1)*360.0/N;
      //  Loop around ring
      for (j=0;j<N;j++)
      {
         float ph0 =  j   *360.0/N;
         float ph1 = (j+1)*360.0/N;
         glNormal3d(Cos(th1)*Cos(ph0),-Sin(th1)*Cos(ph0),Sin(ph0)); glTexCoord2d(th1/30.0,ph0/180.0); glVertex3d(Cos(th1)*(1+r*Cos(ph0)),-Sin(th1)*(1+r*Cos(ph0)),r*Sin(ph0));
         glNormal3d(Cos(th0)*Cos(ph0),-Sin(th0)*Cos(ph0),Sin(ph0)); glTexCoord2d(th0/30.0,ph0/180.0); glVertex3d(Cos(th0)*(1+r*Cos(ph0)),-Sin(th0)*(1+r*Cos(ph0)),r*Sin(ph0));
         glNormal3d(Cos(th0)*Cos(ph1),-Sin(th0)*Cos(ph1),Sin(ph1)); glTexCoord2d(th0/30.0,ph1/180.0); glVertex3d(Cos(th0)*(1+r*Cos(ph1)),-Sin(th0)*(1+r*Cos(ph1)),r*Sin(ph1));
         glNormal3d(Cos(th1)*Cos(ph1),-Sin(th1)*Cos(ph1),Sin(ph1)); glTexCoord2d(th1/30.0,ph1/180.0); glVertex3d(Cos(th1)*(1+r*Cos(ph1)),-Sin(th1)*(1+r*Cos(ph1)),r*Sin(ph1));
      }
   }
   glEnd();

   //  Restore
   glPopMatrix();
}

/*
 *  Draw teapot
 */
static void Teapot(float x,float y,float z , float th,float ph , float S)
{
   //  Transform
   glPushMatrix();
   glTranslated(x,y+0.5,z);
   glRotated(ph,1,0,0);
   glRotated(th,0,1,0);

   //  Draw solid teapot
   glColor3f(0,1,0);
   glutSolidTeapot(2*S);

   //  Restore
   glPopMatrix();
}

/*
 *  Draw a wall
 */
static void Wall(float x,float y,float z, float th,float ph , float Sx,float Sy,float Sz , float St)
{
   int   i,j;
   float s=1.0/n;
   float t=0.5*St/n;

   //  Transform
   glPushMatrix();
   glTranslated(x,y,z);
   glRotated(ph,1,0,0);
   glRotated(th,0,1,0);
   glScaled(Sx,Sy,Sz);

   //  Draw walls
   glNormal3f(0,0,1);
   for (j=-n;j<n;j++)
   {
      glBegin(GL_QUAD_STRIP);
      for (i=-n;i<=n;i++)
      {
         glTexCoord2f((i+n)*t,(j  +n)*t); glVertex3f(i*s,    j*s,-1);
         glTexCoord2f((i+n)*t,(j+1+n)*t); glVertex3f(i*s,(j+1)*s,-1);
      }
      glEnd();
   }

   //  Restore
   glPopMatrix();
}

/*
 *  Set light
 *    light>0 bright
 *    light<0 dim
 *    light=0 off
 */
static void Light(int light)
{
   //  Set light position
   Lpos[0] = 2*Cos(zh);
   Lpos[1] = Ylight;
   Lpos[2] = 2*Sin(zh);
   Lpos[3] = 1;

   //  Enable lighting
   if (light)
   {
      float Low[]  = {0.1,0.1,0.1,1.0};
      float Med[]  = {0.3,0.3,0.3,1.0};
      float High[] = {1.0,1.0,1.0,1.0};
      //  Enable lighting with normalization
      glEnable(GL_LIGHTING);
      glEnable(GL_NORMALIZE);
      //  glColor sets ambient and diffuse color materials
      glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
      glEnable(GL_COLOR_MATERIAL);
      //  Enable light 0
      glEnable(GL_LIGHT0);
      glLightfv(GL_LIGHT0,GL_POSITION,Lpos);
      glLightfv(GL_LIGHT0,GL_AMBIENT,light>0?Med:Low);
      glLightfv(GL_LIGHT0,GL_DIFFUSE,light>0?High:Med);
   }
   else
   {
      glDisable(GL_LIGHTING);
      glDisable(GL_COLOR_MATERIAL);
      glDisable(GL_NORMALIZE);
   }
}

/*
 *  Draw scene
 *    light>0  lit colors
 *    light<0  shaded colors
 *    light=0  shadow volumes
 */
void Scene(int light)
{
   int k;  // Counters used to draw floor

   //  Set light position and properties
   Light(light);
 
   //  Enable textures if lit
   if (MultiTex && light)
   {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D,tex2d[2]);
   }

   //  Draw objects         x    y   z          th,ph    dims   
   if (obj&0x01)     Cube(-0.8,+0.8,0.0 , -0.25*zh, 0  , 0.3    );
   if (obj&0x02) Cylinder(+0.8,+0.5,0.0 ,   0.5*zh,zh  , 0.2,0.5);
   if (obj&0x04)    Torus(+0.5,-0.8,0.0 ,        0,zh  , 0.5,0.2);
   if (obj&0x08)   Teapot(-0.5,-0.5,0.0 ,     2*zh, 0  , 0.25   );

   //  Disable textures
   if (MultiTex && light) glDisable(GL_TEXTURE_2D);

   //  The floor, ceiling and walls don't cast a shadow, so bail here
   if (!light) return;

   //  Always enable textures
   if (MultiTex) glEnable(GL_TEXTURE_2D);

   //  Water texture for floor and ceiling
   if (MultiTex) glBindTexture(GL_TEXTURE_2D,tex2d[0]);
   glColor3f(1.0,1.0,1.0);
   for (k=-1;k<=box;k+=2)
      Wall(0,0,0, 0,90*k , 8,8,box?6:2 , 4);
   //  Crate texture for walls
   if (MultiTex) glBindTexture(GL_TEXTURE_2D,tex2d[1]);
   for (k=0;k<4*box;k++)
      Wall(0,0,0, 90*k,0 , 8,box?6:2,8 , 1);

   //  Disable textures
   if (MultiTex) glDisable(GL_TEXTURE_2D);
}

/*
 *  OpenGL (GLUT) calls this routine to display the scene
 */
void display()
{
   const double len=2.0;
   //  Eye position
   float Ex = -2*dim*Sin(th)*Cos(ph);
   float Ey = +2*dim        *Sin(ph);
   float Ez = +2*dim*Cos(th)*Cos(ph);

   //  Erase the window and the depth buffers
   glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
   //  Disable lighting
   glDisable(GL_LIGHTING);

   //
   //  Draw the scene with shadows
   //
   //  Set perspective view
   if (mode)
   {
      //  Half width for shadow map display
      Project(60,asp/2,dim);
      glViewport(0,0,Width/2,Height);
   }
   else
   {
      //  Full width
      Project(60,asp,dim);
      glViewport(0,0,Width,Height);
   }
   gluLookAt(Ex,Ey,Ez , 0,0,0 , 0,Cos(ph),0);

   //  Draw light position as sphere (still no lighting here)
   glColor3f(1,1,1);
   glPushMatrix();
   glTranslated(Lpos[0],Lpos[1],Lpos[2]);
   glutSolidSphere(0.03,10,10);
   glPopMatrix();

   //  Shadow pass - needed if ambient shadows are not supported
   if (NoAmbient)
   {
      //  Draw scene with dim lighting (including walls)
      Scene(-1);
      // Enable alpha test so that shadowed fragments are discarded
      glAlphaFunc(GL_GREATER,0.9);
      glEnable(GL_ALPHA_TEST);
   }

   // Set up shadow texture comparison
   if (MultiTex) glActiveTexture(GL_TEXTURE1);
   glEnable(GL_TEXTURE_2D);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_COMPARE_MODE,GL_COMPARE_R_TO_TEXTURE);

   // Set up the eye plane for projecting the shadow map on the scene
   glEnable(GL_TEXTURE_GEN_S); glTexGendv(GL_S,GL_EYE_PLANE,Svec);
   glEnable(GL_TEXTURE_GEN_T); glTexGendv(GL_T,GL_EYE_PLANE,Tvec);
   glEnable(GL_TEXTURE_GEN_R); glTexGendv(GL_R,GL_EYE_PLANE,Rvec);
   glEnable(GL_TEXTURE_GEN_Q); glTexGendv(GL_Q,GL_EYE_PLANE,Qvec);
   if (MultiTex) glActiveTexture(GL_TEXTURE0);

   // Draw objects in the scene (including walls)
   Scene(1);

   //  Disable textures and texture generation
   if (MultiTex) glActiveTexture(GL_TEXTURE1);
   glDisable(GL_TEXTURE_2D);
   glDisable(GL_TEXTURE_GEN_S);
   glDisable(GL_TEXTURE_GEN_T);
   glDisable(GL_TEXTURE_GEN_R);
   glDisable(GL_TEXTURE_GEN_Q);
   if (MultiTex) glActiveTexture(GL_TEXTURE0);

   //  Disable alpha test if the shadow pass is done
   if (NoAmbient) glDisable(GL_ALPHA_TEST);
   ErrCheck("display - image");

   //  Draw axes (white)
   glColor3f(1,1,1);
   if (axes)
   {
      glBegin(GL_LINES);
      glVertex3d(0.0,0.0,0.0);
      glVertex3d(len,0.0,0.0);
      glVertex3d(0.0,0.0,0.0);
      glVertex3d(0.0,len,0.0);
      glVertex3d(0.0,0.0,0.0);
      glVertex3d(0.0,0.0,len);
      glEnd();
      //  Label axes
      glRasterPos3d(len,0.0,0.0);
      Print("X");
      glRasterPos3d(0.0,len,0.0);
      Print("Y");
      glRasterPos3d(0.0,0.0,len);
      Print("Z");
   }

   //
   //  Show the shadow map
   //
   if (mode)
   {
      int n,ix=Width/2+5,iy=Height-5;
      //  Aspect ratio of texture
      double rat=(double)Sdim/Tdim;
      //  Texture containing shadow map
      int tex = (MultiTex) ? GL_TEXTURE1 : GL_TEXTURE0;
      glActiveTexture(tex);
      //  Orthogonal view (right half)
      Project(0,asp/2,1);
      glViewport(Width/2+1,0,Width/2,Height);
      //  Disable any manipulation of textures
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_COMPARE_MODE,GL_NONE);
      //  Display texture by drawing quad
      glEnable(GL_TEXTURE_2D);
      glColor3f(1,1,1);
      glBegin(GL_QUADS);
      glMultiTexCoord2f(tex,0,0);glVertex2f(-rat,-1);
      glMultiTexCoord2f(tex,1,0);glVertex2f(+rat,-1);
      glMultiTexCoord2f(tex,1,1);glVertex2f(+rat,+1);
      glMultiTexCoord2f(tex,0,1);glVertex2f(-rat,+1);
      glEnd();
      glDisable(GL_TEXTURE_2D);
      //  Switch back to default texture
      if (MultiTex) glActiveTexture(GL_TEXTURE0);
      //  Show buffer info
      glColor3f(1,0,0);
      glGetIntegerv(GL_MAX_TEXTURE_UNITS,&n);
      glWindowPos2i(ix,iy-=20);
      Print("Maximum Texture Units %d\n",n);
      glGetIntegerv(GL_MAX_TEXTURE_SIZE,&n);
      glWindowPos2i(ix,iy-=20);
      Print("Maximum Texture Size %d\n",n);
      glGetIntegerv(GL_MAX_RENDERBUFFER_SIZE_EXT,&n);
      glWindowPos2i(ix,iy-=20);
      Print("Maximum Buffer Size  %d\n",n);
      glWindowPos2i(ix,iy-=20);
      Print("Shadow Texture Size %dx%d\n",Sdim,Tdim);
      glWindowPos2i(ix,iy-=20);
      if (NoAmbient) Print("Ambient Shadows not supported\n");
      glWindowPos2i(ix,iy-=20);
      if (!MultiTex) Print("Multi-textures not supported\n");
   }

   //  Display parameters
   glColor3f(1,1,1);
   glWindowPos2i(5,5);
   Print("Ylight=%.1f Angle=%d,%d,%d  Dim=%.1f Slices=%d Mode=%s",
     Ylight,th,ph,zh,dim,n,text[mode]);

   //  Render the scene and make it visible
   glFlush();
   glutSwapBuffers();
}

/*
 *  Build Shadow Map
 */
void ShadowMap(void)
{
   double Lmodel[16];  //  Light modelview matrix
   double Lproj[16];   //  Light projection matrix
   double Tproj[16];   //  Texture projection matrix
   double Dim=2.0;     //  Bounding radius of scene
   double Ldist;       //  Distance from light to scene center

   //  Save transforms and modes
   glPushMatrix();
   glPushAttrib(GL_TRANSFORM_BIT|GL_ENABLE_BIT);
   //  No write to color buffer and no smoothing
   glShadeModel(GL_FLAT);
   glColorMask(0,0,0,0);
   // Overcome imprecision
   glEnable(GL_POLYGON_OFFSET_FILL);

   //  Turn off lighting and set light position
   Light(0);

   //  Light distance
   Ldist = sqrt(Lpos[0]*Lpos[0] + Lpos[1]*Lpos[1] + Lpos[2]*Lpos[2]);
   if (Ldist<1.1*Dim) Ldist = 1.1*Dim;

   //  Set perspective view from light position
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   gluPerspective(114.6*atan(Dim/Ldist),(float)Sdim/Tdim,Ldist-Dim,Ldist+Dim);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   gluLookAt(Lpos[0],Lpos[1],Lpos[2] , 0,0,0 , 0,1,0);
   //  Size viewport to desired dimensions
   glViewport(0,0,Sdim,Tdim);

   // Redirect traffic to the frame buffer
   if (framebuf>0) glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,framebuf);

   // Clear the depth buffer
   glClear(GL_DEPTH_BUFFER_BIT);
   // Draw all objects that can cast a shadow
   Scene(0);

   // Copy depth values into depth texture
   if (MultiTex) glActiveTexture(GL_TEXTURE1);
   glCopyTexImage2D(GL_TEXTURE_2D,0,GL_DEPTH_COMPONENT,0,0,Sdim,Tdim,0);
   if (MultiTex) glActiveTexture(GL_TEXTURE0);

   //  Retrieve light projection and modelview matrices
   glGetDoublev(GL_PROJECTION_MATRIX,Lproj);
   glGetDoublev(GL_MODELVIEW_MATRIX,Lmodel);

   // Set up texture matrix for shadow map projection,
   // which will be rolled into the eye linear
   // texture coordinate generation plane equations
   glLoadIdentity();
   glTranslated(0.5,0.5,0.5);
   glScaled(0.5,0.5,0.5);
   glMultMatrixd(Lproj);
   glMultMatrixd(Lmodel);

   // Retrieve result and transpose to get the s, t, r, and q rows for plane equations
   glGetDoublev(GL_MODELVIEW_MATRIX,Tproj);
   Svec[0] = Tproj[0];    Tvec[0] = Tproj[1];    Rvec[0] = Tproj[2];    Qvec[0] = Tproj[3];
   Svec[1] = Tproj[4];    Tvec[1] = Tproj[5];    Rvec[1] = Tproj[6];    Qvec[1] = Tproj[7];
   Svec[2] = Tproj[8];    Tvec[2] = Tproj[9];    Rvec[2] = Tproj[10];   Qvec[2] = Tproj[11];
   Svec[3] = Tproj[12];   Tvec[3] = Tproj[13];   Rvec[3] = Tproj[14];   Qvec[3] = Tproj[15];

   // Restore normal drawing state
   glShadeModel(GL_SMOOTH);
   glColorMask(1,1,1,1);
   glDisable(GL_POLYGON_OFFSET_FILL);
   glPopAttrib();
   glPopMatrix();
   if (framebuf>0) glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);

   //  Check if something went wrong
   ErrCheck("ShadowMap");
}

/*
 *
 */
void InitMap()
{
   unsigned int shadowtex; //  Shadow buffer texture id
   int n;

   //  See if multi-textures are supported
   glGetIntegerv(GL_MAX_TEXTURE_UNITS,&n);
   MultiTex = (n>1);

   //  Get maximum texture buffer size
   glGetIntegerv(GL_MAX_TEXTURE_SIZE,&shadowdim);
   //  Limit texture size to maximum buffer size
   glGetIntegerv(GL_MAX_RENDERBUFFER_SIZE_EXT,&n);
   if (shadowdim>n) shadowdim = n;
   //  Limit texture size to 2048 for performance
   if (shadowdim>2048) shadowdim = 2048;

   //  Do Shadow textures in MultiTexture
   if (MultiTex) glActiveTexture(GL_TEXTURE1);

   //  Allocate and bind shadow texture
   glGenTextures(1,&shadowtex);
   glBindTexture(GL_TEXTURE_2D,shadowtex);

   //  Map single depth value to RGBA (this is called intensity)
   glTexParameteri(GL_TEXTURE_2D,GL_DEPTH_TEXTURE_MODE,GL_INTENSITY);

   //  Check if ambient shadows are supported
   glGetError();
   glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_COMPARE_FAIL_VALUE_ARB,0.5f);
   NoAmbient = (glGetError()==GL_INVALID_ENUM);

   //  Texture modulate underlying objects
   glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);

   //  Set texture mapping to clamp and linear interpolation
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

   //  Set automatic texture generation mode to Eye Linear
   glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_EYE_LINEAR);
   glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_EYE_LINEAR);
   glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_EYE_LINEAR);
   glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_EYE_LINEAR);

   // Switch back to default textures
   if (MultiTex) glActiveTexture(GL_TEXTURE0);

   // Only use framebuffer if it is big enough
   if (shadowdim>512)
   {
      unsigned int renderbuf;
      // Use power of two
      Sdim = Tdim = shadowdim = power2(shadowdim);
      // Set up some renderbuffer state
      glGenFramebuffersEXT(1,&framebuf);
      glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,framebuf);
      //  Set render buffer size to shadow map
      glGenRenderbuffersEXT(1,&renderbuf);
      glBindRenderbufferEXT(GL_RENDERBUFFER_EXT,renderbuf);
      glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT,GL_DEPTH_COMPONENT32,shadowdim,shadowdim);
      glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,GL_DEPTH_ATTACHMENT_EXT,GL_RENDERBUFFER_EXT,renderbuf);
      //  Don't write or read to visible color buffer
      glDrawBuffer(GL_NONE);
      glReadBuffer(GL_NONE);
      //  Make sure this all worked
      if (glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT) != GL_FRAMEBUFFER_COMPLETE_EXT) Fatal("Error setting up frame buffer\n");
      glBindFramebufferEXT(GL_FRAMEBUFFER_EXT,0);

      //  Create shadow map
      ShadowMap();
   }
   else
   {
      //  Set shadow dim to maximum
      glGetIntegerv(GL_MAX_TEXTURE_SIZE,&shadowdim);
      printf("Insufficient framebuffer\n");
   }

   //  Check if something went wrong
   ErrCheck("InitMap");
}

/*
 *  GLUT calls this routine when nothing else is going on
 */
void idle(int k)
{
   //  Elapsed time in seconds
   double t = glutGet(GLUT_ELAPSED_TIME)/1000.0;
   zh = fmod(90*t,1440.0);
   //  Update shadow map
   ShadowMap();
   //  Tell GLUT it is necessary to redisplay the scene
   glutPostRedisplay();
   //  Schedule update
   if (move) glutTimerFunc(dt,idle,0);
}

/*
 *  GLUT calls this routine when an arrow key is pressed
 */
void special(int key,int x,int y)
{
   //  Right arrow key - increase angle by 5 degrees
   if (key == GLUT_KEY_RIGHT)
      th += 1;
   //  Left arrow key - decrease angle by 5 degrees
   else if (key == GLUT_KEY_LEFT)
      th -= 1;
   //  Up arrow key - increase elevation by 5 degrees
   else if (key == GLUT_KEY_UP)
      ph += 1;
   //  Down arrow key - decrease elevation by 5 degrees
   else if (key == GLUT_KEY_DOWN)
      ph -= 1;
   //  PageUp key - increase dim
   else if (key == GLUT_KEY_PAGE_DOWN)
      dim += 0.1;
   //  PageDown key - decrease dim
   else if (key == GLUT_KEY_PAGE_UP && dim>1)
      dim -= 0.1;
   //  Keep angles to +/-360 degrees
   th %= 360;
   ph %= 360;
   //  Tell GLUT it is necessary to redisplay the scene
   glutPostRedisplay();
}

/*
 *  GLUT calls this routine when a key is pressed
 */
void key(unsigned char ch,int x,int y)
{
   //  Exit on ESC
   if (ch == 27)
      exit(0);
   //  Reset view angle
   else if (ch == '0')
      th = ph = 0;
   //  Toggle axes
   else if (ch == 'a' || ch == 'A')
      axes = 1-axes;
   //  Toggle display modes
   else if (ch == 'm' || ch == 'M')
      mode = 1-mode;
   //  Toggle light movement
   else if (ch == 's' || ch == 'S')
      move = 1-move;
   //  Toggle box
   else if (ch == 'b' || ch == 'B')
      box = 1-box;
   //  Toggle objects
   else if (ch == 'o')
      obj = (obj+1)%16;
   else if (ch == 'O')
      obj = (obj+15)%16;
   //  Light elevation
   else if (ch=='-')
      Ylight -= 0.1;
   else if (ch=='+')
      Ylight += 0.1;
   //  Light azimuth
   else if (ch=='[')
      zh -= 1;
   else if (ch==']')
      zh += 1;
   //  Number of patches
   else if (ch=='<' && n>1)
      n--;
   else if (ch=='>' && n<MAXN)
      n++;
   //  Restart animation
   if ((ch =='s' || ch == 'S') && move) glutTimerFunc(dt,idle,0);
   //  Update screen size when mode changes
   if (ch == 'm' || ch == 'M') glutReshapeWindow(mode?2*Width:Width/2,Height);
   //  Update shadow map if light position or objects changed
   if (strchr("<>oO-+[]",ch)) ShadowMap();
   //  Tell GLUT it is necessary to redisplay the scene
   glutPostRedisplay();
}

/*
 *  GLUT calls this routine when the window is resized
 */
void reshape(int width,int height)
{
   //  Ratio of the width to the height of the window
   asp = (height>0) ? (double)width/height : 1;
   //  Store window dimensions
   Width  = width;
   Height = height;
   //  If frame buffer is not used, resize texture and regenerate
   if (framebuf==0)
   {
      Sdim = power2((width  < shadowdim) ? width  : shadowdim);
      Tdim = power2((height < shadowdim) ? height : shadowdim);
      ShadowMap();
   }
}

/*
 *  Start up GLUT and tell it what to do
 */
int main(int argc,char* argv[])
{
   //  Initialize GLUT
   glutInit(&argc,argv);
   //  Request double buffered, true color window with Z buffering & stencil at 600x600
   glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);
   glutInitWindowSize(600,600);
   glutCreateWindow("Shadow Map");
#ifdef USEGLEW
   //  Initialize GLEW
   if (glewInit()!=GLEW_OK) Fatal("Error initializing GLEW\n");
#endif
   //  Set callbacks
   glutDisplayFunc(display);
   glutReshapeFunc(reshape);
   glutSpecialFunc(special);
   glutKeyboardFunc(key);
   glutTimerFunc(dt,idle,0);
   //  Load textures
   tex2d[0] = LoadTexBMP("water.bmp");
   tex2d[1] = LoadTexBMP("crate.bmp");
   tex2d[2] = LoadTexBMP("pi.bmp");
   // Enable Z-buffer
   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LEQUAL);
   glPolygonOffset(4,0);
   //  Initialize texture map
   InitMap();
   //  Pass control to GLUT so it can interact with the user
   glutMainLoop();
   return 0;
}
