/*
 *  Mandelbulb code for both CPU and GPU usage
 *
 *  define DEVICE as null for CPU and __device__ for CUDA
 */
#define Rmax    2
#define MaxDist 4
#define MaxStep 100

/*
 *  Mandlebulb distance estimate
 */
DEVICE inline float Step(const Vec3& C,const MandelBulb& mb)
{
   //  Skip one iteration (Z = 0 + C)
   Vec3  Z  = C;
   //  Compute |Z|
   float R  = sqrtf(Z.x*Z.x+Z.y*Z.y+Z.z*Z.z);
   //  Initialize |Z'|
   float dR = 1;
   //  Loop maxiter times or until divergence
   int   i = mb.maxiter;
   while (R<Rmax && i--)
   {
      //  Convert to spherical
      float ph = asinf(Z.z/R);
      float th = atan2f(Z.y,Z.x);
      //  Compute derivative recursively n|Z^(n-1)|Z'
      float Rn = powf(R,mb.n-1);
      dR = Rn*dR*mb.n + 1;
      //  Compute R^n
      Rn *= R;
      //  Compute sin(n*ph) & cos(n*ph)
      float sinph,cosph;
      sincosf(mb.n*ph,&sinph,&cosph);
      //  Compute sin(n*th) & cos(n*th)
      float sinth,costh;
      sincosf(mb.n*th,&sinth,&costh);
      //  Compute Z^n+C
      Z = Rn*Vec3(cosph*costh,cosph*sinth,sinph) + C;
      //  Compute |Z|
      R = sqrtf(Z.x*Z.x+Z.y*Z.y+Z.z*Z.z);
   }
   //  Evaluate G(c)/G'(c)
   return 0.5f*logf(R)*R/dR;
}

/*
 *  Mandelbulb normal vector
 */
DEVICE inline Vec3 Normal(const Vec3 &C,const MandelBulb& mb)
{
   const float d = 1e-3;
   if (mb.fast)
   {
      float c  = Step(C,mb);
      float cx = Step(C+Vec3(d,0,0),mb);
      float cy = Step(C+Vec3(0,d,0),mb);
      float cz = Step(C+Vec3(0,0,d),mb);
      return normalize(Vec3(c-cx,c-cy,c-cz));
   }
   else
   {
      float cx = Step(C-Vec3(d,0,0),mb) - Step(C+Vec3(d,0,0),mb);
      float cy = Step(C-Vec3(0,d,0),mb) - Step(C+Vec3(0,d,0),mb);
      float cz = Step(C-Vec3(0,0,d),mb) - Step(C+Vec3(0,0,d),mb);
      return normalize(Vec3(cx,cy,cz));
   }
}

/*
 *  Adaptive ray marching
 */
DEVICE inline bool RayMarchA(const Vec3& org, const Vec3& dir,const MandelBulb& mb,Vec3& point)
{
   const float eps = 1e-3;
   float dist  = 0;
   point = org;
   int k=MaxStep;
   //  Estimate distance until max distance or max steps
   while (dist<MaxDist && k--)
   {
      float step = Step(point,mb);
      point += step*dir;
      dist  += step;
      //  Stop of step is small
      if (fabs(step)<eps) return true;
   }
   return false;
}

/*
 *  Non-adaptive ray marching
 */
DEVICE inline bool RayMarchNA(Vec3 point,const Vec3& dir,const MandelBulb& mb)
{
   if (!mb.shadow) return false;
   float step = 0.1;
   float dist = 0;
   const float eps = 0.1*step;
   int k=MaxStep;
   //  Estimate distance until max distance or max steps
   while (dist<MaxDist && k--)
   {
      point += step*dir;
      dist  += step;
      step *= 1.1f; //increase step regularly (ie resolution decrease with distance)
      if (Step(point,mb)<eps) return true;
   }
   return false;
}

/*
 *  Ray trace a pixel
 */
DEVICE Color RayTrace(Vec3 org,Vec3 dir,const MandelBulb& mb)
{
   //  Scalar intensity
   float f=0;
   Vec3  P;
   //  Test for intersection
   if (RayMarchA(org,dir,mb,P))
   {
      //  Add ambient
      f = 0.3;
      //  Compute normal vector
      Vec3 N = Normal(P,mb);
      //  Compute cosine with light vector
      float I = -(mb.L*N);
      //  If illuminated and not in shadow, add diffuse
      if (I>0 && !RayMarchNA(P,mb.L,mb)) f += 0.7*I;
   }
   //  Return the resulting color
   return Color(f,f,0);
}

/*
 *  Ray trace a pixel
 */
DEVICE void RayTracePixel(unsigned char* pix,int k,const MandelBulb& mb)
{
   //  Translate scalar index into pixel
   int i = k%mb.wid;
   int j = k/mb.wid;
   // Cast the intial ray from (x,y,1.1) down z axis
   Vec3 org(mb.zoom*(i-mb.wid/2) , mb.zoom*(j-mb.hgt/2) , 1.1);
   Vec3 dir(0,0,-1);
   Color col = RayTrace(mb.rot*org , mb.rot*dir , mb);
   //  Copy color to pixel array
   pix += 4*k;
   *pix++ = col.r>1 ? 255 : (unsigned int)(255*col.r);
   *pix++ = col.g>1 ? 255 : (unsigned int)(255*col.g);
   *pix++ = col.b>1 ? 255 : (unsigned int)(255*col.b);
   *pix++ = 255;
}

/*
 *  Convenience routine to output raster text
 *  Use VARARGS to make this more flexible
 */
#define LEN 8192  //  Maximum length of text string
void Print(const char* format , ...)
{
   char    buf[LEN];
   char*   ch=buf;
   va_list args;
   //  Turn the parameters into a character string
   va_start(args,format);
   vsnprintf(buf,LEN,format,args);
   va_end(args);
   //  Display the characters one at a time at the current raster position
   while (*ch)
      glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18,*ch++);
}

/*
 *  Set rotation matrix
 */
void SetRot(void)
{
   double M[16];
   //  Initialize Modelview
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   //  Apply rotations
   glRotated(-ph,1,0,0);
   glRotated(th,0,1,0);
   //  Copy matrix to row vectors
   glGetDoublev(GL_MODELVIEW_MATRIX,M);
   mb.rot.x.x = M[0]; mb.rot.x.y = M[4]; mb.rot.x.z = M[8];
   mb.rot.y.x = M[1]; mb.rot.y.y = M[5]; mb.rot.y.z = M[9];
   mb.rot.z.x = M[2]; mb.rot.z.y = M[6]; mb.rot.z.z = M[10];
   //  Reset
   glLoadIdentity();
}

/*
 *  GLUT calls this routine when an arrow key is pressed
 */
void special(int key,int x,int y)
{
   //  Right arrow key - increase angle by 5 degrees
   if (key == GLUT_KEY_RIGHT)
      th += 5;
   //  Left arrow key - decrease angle by 5 degrees
   else if (key == GLUT_KEY_LEFT)
      th -= 5;
   //  Up arrow key - increase elevation by 5 degrees
   else if (key == GLUT_KEY_UP)
      ph += 5;
   //  Down arrow key - decrease elevation by 5 degrees
   else if (key == GLUT_KEY_DOWN)
      ph -= 5;
   //  Page Up key - increase zoom
   else if (key == GLUT_KEY_PAGE_UP)
      mb.zoom *= 0.9;
   //  Page Down key - decrease zoom
   else if (key == GLUT_KEY_PAGE_DOWN)
      mb.zoom *= 1.1;
   //  Keep angles to +/-360 degrees
   th %= 360;
   ph %= 360;
   SetRot();
   //  Tell GLUT it is necessary to redisplay the scene
   glutPostRedisplay();
}

/*
 *  GLUT calls this routine when a key is pressed
 */
void key(unsigned char ch,int x,int y)
{
   //  Exit on ESC
   if (ch == 27)
      exit(0);
   //  Reset view angle
   else if (ch == '0')
      th = ph = 0;
   //  Change power
   else if (ch == '-' && mb.n>1)
      mb.n--;
   else if (ch == '+')
      mb.n++;
   //  Change maximum iterations
   else if (ch == '[' && mb.maxiter>0)
      mb.maxiter--;
   else if (ch == ']')
      mb.maxiter++;
   //  Increase angle by 5 degrees
   else if (ch == 'a')
      Th += 5;
   //  Decrease angle by 5 degrees
   else if (ch == 'd')
      Th -= 5;
   //  Increase elevation by 5 degrees
   else if (ch == 'w')
      Ph += 5;
   //  Decrease elevation by 5 degrees
   else if (ch == 's')
      Ph -= 5;
   //  Decrease elevation by 5 degrees
   else if (ch == ' ')
      mb.shadow = !mb.shadow;
   else if (ch == '.')
      mb.fast = !mb.fast;
   //  Reset Angle
   SetRot();
   //  Tell GLUT it is necessary to redisplay the scene
   glutPostRedisplay();
}

/*
 *  GLUT calls this routine when the window is resized
 */
void reshape(int width,int height)
{
   //  Window dimensions
   mb.wid = width;
   mb.hgt = height;
   mb.dim = 4*width*height;
   //  Allocate pixels size of window
   delete pixels;
   pixels = new unsigned char [mb.dim];
   //  Set the viewport to the entire window
   glViewport(0,0, width,height);
}

/*
 *  Initialize
 */
void Init(MandelBulb& mb)
{
   //  Mandelbulb parameters
   mb.n=6;             //  Power
   mb.maxiter=10;      //  Max iterations
   mb.shadow=true;     //  Shadows
   mb.fast=true;       //  Fast normals
   mb.zoom=0.005;      //  Zoom level
   //  Projection
   SetRot();
}
